﻿#!/usr/bin/env ruby -w
# encoding: utf-8

require 'test/unit'
require 'yaml'
require 'set'

def dump_yaml_a file_name, obj
  IO.write(file_name, YAML.dump(obj), :encoding => 'utf-8')
end

def load_yaml_a file_name
  YAML.load(IO.read(file_name, :encoding => "bom|utf-8"))
end

def dump_yaml_b filename, obj
  File.open(filename, 'w:utf-8') { |fil| YAML.dump(obj, fil) }
end

def load_yaml_b filename
  File.open(filename, 'r:bom|utf-8') { |fil| return YAML.load(fil) }
end

def dump_yaml_x file_name, obj
  IO.write(file_name, obj.to_yaml, :encoding => 'utf-8')
end

def dump_marshal_a filename, obj
  IO.write(filename, Marshal.dump(obj), :encoding => 'binary')
end

def load_marshal_a filename
  Marshal.load(IO.read(filename, :encoding => 'binary'))
end

def dump_marshal_b filename, obj
  File.open(filename, 'wb') { |fil| Marshal.dump(obj, fil) }
end

def load_marshal_b filename
  File.open(filename, 'rb') { |fil| return Marshal.load(fil) }
end

class LoadDumpTest < Test::Unit::TestCase
  TEST_FILE_A = "test_file.a"
  TEST_FILE_B = "test_file.b"
  TEST_FILE_X = "test_file.x"

  def yaml_load_dump_same obj
    dump_yaml_a TEST_FILE_A, obj
    dump_yaml_b TEST_FILE_B, obj
    obj_aa = load_yaml_a TEST_FILE_A
    obj_bb = load_yaml_b TEST_FILE_B
    assert_equal obj_aa, obj_bb
  end

  def yaml_load_dump_cross obj
    dump_yaml_a TEST_FILE_A, obj
    dump_yaml_b TEST_FILE_B, obj
    dump_yaml_x TEST_FILE_X, obj
    obj_aa = load_yaml_a TEST_FILE_A
    obj_ab = load_yaml_a TEST_FILE_B
    obj_ba = load_yaml_b TEST_FILE_A
    obj_bb = load_yaml_b TEST_FILE_B
    obj_ax = load_yaml_a TEST_FILE_X
    obj_bx = load_yaml_b TEST_FILE_X
    assert_equal obj_aa, obj_ab
    assert_equal obj_ab, obj_ba
    assert_equal obj_ba, obj_bb
    assert_equal obj_bb, obj_ax
    assert_equal obj_ax, obj_bx
  end

  def marshal_load_dump_same obj
    dump_marshal_a TEST_FILE_A, obj
    dump_marshal_b TEST_FILE_B, obj
    obj_aa = load_marshal_a TEST_FILE_A
    obj_bb = load_marshal_b TEST_FILE_B
    assert_equal obj_aa, obj_bb
  end

  def marshal_load_dump_cross obj
    dump_marshal_a TEST_FILE_A, obj
    dump_marshal_b TEST_FILE_B, obj
    obj_aa = load_marshal_a TEST_FILE_A
    obj_ab = load_marshal_a TEST_FILE_B
    obj_ba = load_marshal_b TEST_FILE_A
    obj_bb = load_marshal_b TEST_FILE_B
    assert_equal obj_aa, obj_ab
    assert_equal obj_ab, obj_ba
    assert_equal obj_ba, obj_bb
  end

  def test_load_dump_1
    obj = %w[Chinese English]
    yaml_load_dump_same obj
    yaml_load_dump_cross obj
    marshal_load_dump_same obj
    marshal_load_dump_cross obj
  end

  def test_load_dump_2
    obj = {:cn=>'Chinese', :en=>'English'}
    yaml_load_dump_same obj
    yaml_load_dump_cross obj
    marshal_load_dump_same obj
    marshal_load_dump_cross obj
  end

  def test_load_dump_3
    obj = {:array=>%w[Chinese English], :hash=>{:c=>'Chinese', :e=>'English'}}
    yaml_load_dump_same obj
    yaml_load_dump_cross obj
    marshal_load_dump_same obj
    marshal_load_dump_cross obj
  end

  def test_load_dump_4
    obj = Set.new(%w[Chinese English])
    yaml_load_dump_same obj
    yaml_load_dump_cross obj
    marshal_load_dump_same obj
    marshal_load_dump_cross obj
  end
end
