require 'uri'
require 'stringio'

module LazyRack
	##
	# Server for a single request, performed in a ractor.
	class RequestServer
		def initialize(sock, addr, app)
			@sock = sock
			@addr = addr
			@app = app
		end

		CRLF = "\r\n".freeze
		VERSION = "HTTP/1.1".freeze

    def serve!
      env = parse_req

      status, headers, body = @app.call(env)

      @sock.write "#{VERSION} #{status}"
      @sock.write(CRLF)
      @sock.write("Connection: close#{CRLF}")

      headers.each do |k, v|
        @sock.write(k)
        @sock.write(" ")
        @sock.write(v)
        @sock.write(CRLF)
      end

      @sock.write(CRLF)

      body.each { |c| @sock.write(c) }

      @sock.close
    end

		def parse_req
			matches = /\A(?<method>\S+)\s+(?<uri>\S+)\s+(?<version>\S+)#{CRLF}\Z/.match(@sock.gets)

			env = {
				"rack.errors" => $stderr,
				"rack.version" => [2, 3],
				"rack.url_scheme" => "http",
				"REQUEST_METHOD" => matches[:method],
				"REQUEST_URI" => matches[:uri],
				"HTTP_VERSION" => matches[:version],
				"QUERY_STRING" => "",
				"SERVER_PORT" => 8000,
				"SERVER_NAME" => "localhost",
				"PATH_INFO" => "",
				"SCRIPT_NAME" => "",
			}

			while matches = /\A(?<key>[^:]+):\s*(?<value>.+)#{CRLF}\Z/.match(hl = @sock.gets)
				case matches[:key]
				when /Content-Type/ then env["CONTENT_TYPE"] = matches[:value]
				when /Content-Length/ then env["CONTENT_LENGTH"] = Integer(matches[:value])
				else env["HTTP_" + matches[:key].tr("-", "_").upcase] ||= matches[:value]
				end
			end

      env["rack.input"] = StringIO.new(@sock.read(env["CONTENT_LENGTH"] || 0))

      env
		end
	end
end
