require 'minitest/autorun'

class ThreadPool
  attr_accessor :size, :threads, :jobs

  def initialize(size)
    @size    = size
    @jobs    = Queue.new
    @threads = create_threads
  end

  def create_threads
    @size.times.map do |i|
      Thread.start do
        Thread.current[:id] = i

        catch(:thread_quit) do
          loop do
            job, args = @jobs.pop
            job.call *args
          end
        end
      end
    end
  end

  def join
    threads.each &:join
  end

  def shutdown
    size.times do
      add { throw :thread_quit }
    end

    threads.map &:join
  end

  def add(*args, &job)
    jobs << [job, args]
  end

  def queue_size;
    jobs.size
  end
end

describe ThreadPool do
  before do
    @pool = ThreadPool.new 2
    @pool.extend MiniTest::Expectations
  end

  it 'generates "No live threads left. Deadlock?deadlock?"' do
    100.times do |i|
      @pool.add(i) do |i|
        puts "doing (#{i + 1})..."
        sleep 0.01
      end
    end

    begin
      @pool.join
    rescue Exception => e
      assert_equal 'No live threads left. Deadlock?', e.message
    end
  end

  it 'works with workaround' do
    100.times do |i|
      @pool.add(i) do |i|
        puts "doing (#{i + 1})..."
        sleep 0.01
      end
    end

    exception = nil

    begin
      Thread.start do
        break until @pool.threads.select(&:alive?).size.zero?
      end.join
      sleep 1
    rescue Exception => e
      exception = e
    end

    assert_nil exception
  end
end