#!/usr/bin/ruby -Ku
#-*- encoding :utf-8 -*-
# -Ku は非推奨ですが。下位互換のために付与
$LOAD_PATH << File.dirname(__FILE__)

=begin
  Ruby 1.8.7-p375
  Ruby 1.9.3-p545
  Ruby 2.1.1-p76

  いずれのバージョンでも、同様のコアダンプを発生させています。
  Ruby 2.1.1-p76 で実行時のコアダンプを添付します。
  
  実行環境は
  Ubuntu 13.10 x86_64 です。CPU の Byte Order は Little Endian です。
  
=end


# Ruby バージョン毎に処理を分岐。
case
  # Ruby 1.9, 1.8 では 'dl/import'
  when RUBY_VERSION.match(/1\.9/), RUBY_VERSION.match(/1\.8\.7/)
    require 'dl/import'
  # Ruby 2.x では 'fiddle/import'
  when RUBY_VERSION.match(/2\.\d/)
    require 'fiddle/import'
end

module LibAquesTalk2
  
  # Ruby バージョン毎に処理を分岐。
  case
    # Ruby 1.9, 1.8.7 では DL::Importer
    when RUBY_VERSION.match(/1\.9/), RUBY_VERSION.match(/1\.8\.7/)
      extend DL::Importer
    # Ruby 2.x では Fiddle::Importer
    when RUBY_VERSION.match(/2\.\d/)
      extend Fiddle::Importer
    # Ruby 1.8.7 older では DL::Importable
    else
      extend DL::Importable
  end
  
  # AquesTalk2 評価版(linux)
  dlload './libAquesTalk2Eva.so.2.3'

  # AquesTalk2 開発用(linux) ←開発用でも同様のコアダンプが発生
  # dlload './libAquesTalk2Eva.so.2.3'


  # AquesTalk2_Synthe_Utf8(koe, iSpeed, size, phontDat)
  # 音声変換を行う処理
  @@synthe = extern('unsigned char * AquesTalk2_Synthe_Utf8(const char *, int, int *, void *)')

  # AquesTalk2_FreeWave(*wav)
  # メモリ解放を行う処理
  @@free_wave = extern('void AquesTalk2_FreeWave(unsigned char *)')
end

# エラークラス
class AquesTalk2_Error < StandardError; end

# AquesTalk2 ライブラリのラッパークラス
class AquesTalk2
  include LibAquesTalk2

  # 読み上げ速度の初期値
  DEFAULT_ISPEED =         100
  # 音声話者の初期値
  DEFAULT_PHONT  =           0
  # 出力ファイル名の初期値
  DEFAULT_OUTPUT = 'output.wav'
  
  # 読み上げ速度アクセサ
  attr_accessor :ispeed
  # 読み上げ話者アクセサ
  attr_accessor :phont
  # 出力ファイル名アクセサ
  attr_accessor :output

  # 引数なしでインスタンス生成された場合、初期値を用いる
  def initialize(ispeed = nil, phont = nil)
    @ispeed ||= DEFAULT_ISPEED
    @phont  ||= DEFAULT_PHONT
  end
  
  # テキストを音声へ変換する処理( AquesTalkライブラリのラッパーメソッド )
  def synthe(input, output = nil, ispeed = nil, phont = nil)
    begin
      raise(AquesTalk2_Error.new('input arg required.')) unless input
    rescue AquesTalk2_Error => error
      return false
    end
    
    input.chomp!
    output ||= DEFAULT_OUTPUT
    ispeed ||= @ispeed
    phont  ||= @phont

    begin
      result = @@synthe.call(input, ispeed, size = 0, phont) # ←このコードが実行された時点でコアダンプ
      raise(AquesTalk2_Error.new('AquesTalk2_Synthe_Utf8 error: '+size.to_s)) if size.zero?
    rescue AquesTalk2_Error => error
      return false
    end
    
    open(output, 'w+b'){|f|f.print(result)}
    free_wave(result)
    return true
  end
  
  # メモリ解放を行う処理のラッパーメソッド
  def free_wave(wav)
    @@free_wave.call(wav)
  end
end

=begin
ここから、テスト処理です。
=end

# 評価用のテキストソース
sample_source = 'こんにちは'
# 出力ファイル名
sample_file = 'sample.wav'

# インスタンス生成
aques_talk = AquesTalk2.new
# テキストから音声ファイルを生成(AquesTalk2 ライブラリを実行)
aques_talk.synthe(sample_source, sample_file)


